#include "stdafx.h"
#include "rwexp.h"
#include "rwcommon.h"
#include "bspexp.h"
#include "bspdlg.h"
#include "dffexp.h"
#include "dffdlg.h"


#include "utilities.h"
#include "modstack.h"
#include "iparamb2.h"

int
DegenerateFace(Mesh *mesh, int face, float vertexWeldThreshold, BOOL weldVertices)
{
    Point3 *v0, *v1, *v2;
    if ((mesh->faces[face].v[0] == mesh->faces[face].v[1]) || 
        (mesh->faces[face].v[0] == mesh->faces[face].v[2]) || 
        (mesh->faces[face].v[1] == mesh->faces[face].v[2]))
    {
        return TRUE;
    }
    else
    {
        if (!weldVertices)
        {
            return FALSE;
        }
        v0 = mesh->getVertPtr(mesh->faces[face].v[0]);
        v1 = mesh->getVertPtr(mesh->faces[face].v[1]);
        v2 = mesh->getVertPtr(mesh->faces[face].v[2]);
        if (((v0->x - v1->x < vertexWeldThreshold) && (v0->x - v1->x > -vertexWeldThreshold) &&
             (v0->y - v1->y < vertexWeldThreshold) && (v0->y - v1->y > -vertexWeldThreshold) &&
             (v0->z - v1->z < vertexWeldThreshold) && (v0->z - v1->z > -vertexWeldThreshold)) ||
            ((v0->x - v2->x < vertexWeldThreshold) && (v0->x - v2->x > -vertexWeldThreshold) &&
             (v0->y - v2->y < vertexWeldThreshold) && (v0->y - v2->y > -vertexWeldThreshold) &&
             (v0->z - v2->z < vertexWeldThreshold) && (v0->z - v2->z > -vertexWeldThreshold)) ||
            ((v1->x - v2->x < vertexWeldThreshold) && (v1->x - v2->x > -vertexWeldThreshold) &&
             (v1->y - v2->y < vertexWeldThreshold) && (v1->y - v2->y > -vertexWeldThreshold) &&
             (v1->z - v2->z < vertexWeldThreshold) && (v1->z - v2->z > -vertexWeldThreshold)))
        {             
            return TRUE;
        }
    }

    return FALSE;
}

int
NumNonDegenerateFaces(Mesh *mesh, Mtl *mat, float vertexWeldThreshold, BOOL export2SidedMaterials, BOOL weldVertices)
{
    int face;
    int numFaces;
    int numNonDegenerateFaces = 0;

    assert (mesh);

    /* calculate number of non-degenerate faces */
    numFaces = mesh->getNumFaces();
    for (face = 0; face < numFaces; face++)
    {
        if (!DegenerateFace(mesh, face, vertexWeldThreshold, weldVertices))
        {        
            numNonDegenerateFaces++;
            if (export2SidedMaterials &&
                DoubleSidedMaterial(mat, mesh->faces[face].getMatID()))
            {
                numNonDegenerateFaces++;
            }            
        }
    }

    return (numNonDegenerateFaces);
}

void
getMtlTiling( Mtl *mat, int MatID,
              bool *UTiling, bool *VTiling, bool *UMirror, bool *VMirror )
{
    *UTiling=false; *VTiling=false; *UMirror=false; *VMirror=false;
    
    Texmap *tmap = GetStdMatTexmap(GetSubMaterial(mat, MatID), ID_DI);
    if (tmap)
    {
        int TileMirror;

        if (tmap->ClassID() == Class_ID(BMTEX_CLASS_ID, 0)) {            
            TileMirror=tmap->GetTextureTiling();
            *UTiling=(TileMirror&U_WRAP)!=0;
            *VTiling=(TileMirror&V_WRAP)!=0;
            *UMirror=(TileMirror&U_MIRROR)!=0;
            *VMirror=(TileMirror&V_MIRROR)!=0;
        }
    }
}

BOOL
isMtlCropped(Mtl *mat, int MatID)
{
    Texmap *tmap = GetStdMatTexmap(GetSubMaterial(mat, MatID), ID_DI);

    if (tmap)
    {
        float U, V, W, H;
        Interval i;

        if (tmap->ClassID() == Class_ID(BMTEX_CLASS_ID, 0)) {
            BitmapTex *bmt = (BitmapTex*) tmap;
            IParamBlock2 *pb = (IParamBlock2 *)bmt->GetReference(1);

            pb->GetValue(0,0,U,i); // U
            pb->GetValue(1,0,V,i); // V
            pb->GetValue(2,0,W,i); // W
            pb->GetValue(3,0,H,i); // H
            if (U != 0.0f || V != 0.0f ||
                W != 1.0f || H != 1.0f)
            {
                return TRUE;
            }
        }
        return FALSE;
    }
    return FALSE;
}

void
ApplyCroppingToUVs(Mtl *mat, int MatID, UVVert *UVs)
{
    Texmap *tmap = GetStdMatTexmap(GetSubMaterial(mat, MatID), ID_DI);

    if (tmap)
    {
        float U, V, W, H, frac;
        Interval i;

        if (tmap->ClassID() == Class_ID(BMTEX_CLASS_ID, 0)) {            
            BitmapTex *bmt = (BitmapTex*) tmap;
            IParamBlock2 *pb = (IParamBlock2 *)bmt->GetReference(1);
            
            pb->GetValue(0,0,U,i); // U
            pb->GetValue(1,0,V,i); // V
            pb->GetValue(2,0,W,i); // W
            pb->GetValue(3,0,H,i); // H

            /* Apply a cropping to uv's */
            frac = UVs->x;
            if (frac < 0.0f)
            {
                frac = (1.0f-frac) - ((1.0f-frac) * W);
            }
            else
            {
                frac -= frac * W;
            }
            UVs->x = (UVs->x - frac) + U;

            frac = UVs->y;
            if (frac < 0.0f)
            {
                frac = (1.0f-frac) - ((1.0f-frac) * H);
            }
            else
            {
                frac -= frac * H;
            }
            UVs->y = (UVs->y - frac) + (1.0f - V - H);
        }    
    }
}

void
GetMaxUVTransform(Mtl *mat, int MatID, Matrix3 *UVTransform)
{
    Texmap *map = GetStdMatTexmap(GetSubMaterial(mat, MatID), ID_DI);

    if (map)
        map->GetUVTransform(*UVTransform);
    else
        UVTransform->IdentityMatrix();        
}


BOOL
CalcTextureOffsets(Mesh *mesh, float *uOffset, float *vOffset, Mtl *mat)
{
    assert(mesh);
    assert(uOffset);
    assert(vOffset);

    /* work out which texture coordinates are actually used by the model */
    int faceInd;
    int numFaces = mesh->getNumFaces();
    int pointIndex;
    RwBool allCropped = TRUE;
    float uMin = FLT_MAX;
    float vMax = FLT_MIN;
    UVVert texVert;
    for (faceInd = 0; faceInd < numFaces; faceInd++)
    {
        TVFace *texFace = &mesh->tvFace[faceInd];
        Matrix3 UVTransform;
        int MatID = mesh->getFaceMtlIndex(faceInd);

        /* only check uncropped materials since all cropped materials
           will be shifted into the 0.0->1.0 UV space */
        if (!isMtlCropped(mat, MatID))
        {
            allCropped = FALSE;
            GetMaxUVTransform(mat, MatID, &UVTransform);

            for (pointIndex = 0; pointIndex < 3; pointIndex++)
            {
                /* Get UV for this next 'used' texVert */
                texVert = mesh->getTVert(texFace->t[pointIndex]);
                /*  Working with 2d mapping so zero z so it can't screw the
                    vector transform */
                texVert.z = 0.0f;

                UVVert transformedUV = texVert * UVTransform;

                ApplyCroppingToUVs(mat, MatID, &transformedUV);

                /* Make sure they're in a valid range */
                if (transformedUV.x < uMin)
                {
                    uMin = transformedUV.x;
                }
    
                if (transformedUV.y > vMax)
                {
                    vMax = transformedUV.y;
                }
            }
        }
    }                    

    if (allCropped)
    {
        *uOffset = 0.0f;
        *vOffset = 1.0f;
    }
    else
    {
        /* used to normalize coordinates */
        *uOffset = (float)(-floor(uMin));
        if ((vMax - floor(vMax)) < 0.001f)
        {
            *vOffset = vMax;
        }
        else
        {
            *vOffset = (float)(vMax + (1 - (vMax - floor(vMax))));
        }
    }

    return (TRUE);
}

BOOL
NodeHasAncestor(INode *node, INode *ancestor)
{    
    while (!node->IsRootNode())
    {
        node = node->GetParentNode();
        if (node == ancestor)
        {
            return true;
        }
    }

    return false;
}


INode *
findNodeByName(INode *root, char *name)
{
    if (strcmp(root->GetName(), name) == 0)
    {
        return root;
    }
    else
    {
        int i;
        int numChildren = root->NumberOfChildren();
        for (i=0; i<numChildren; i++)
        {
            INode *child = root->GetChildNode(i);
            INode *result = NULL;

            result = findNodeByName(child, name);

            if (result)
            {
                return (result);
            }
        }
    }

    return NULL;
}

//iterate through this node's modifier stack.
//return a pointer to a modifier with given class ID if it exists,
//NULL if it doesn't
extern Modifier *
FindModifier( INode *nodePtr, const Class_ID &type )
{
	/* Get object from node. Abort if no object. */
	Object* ObjectPtr = nodePtr->GetObjectRef();			

	if (!ObjectPtr) return NULL;

	/* Is derived object ? */
	while (ObjectPtr->SuperClassID() == GEN_DERIVOB_CLASS_ID && ObjectPtr)
	{
		/* Yes -> Cast. */
		IDerivedObject *DerivedObjectPtr = (IDerivedObject *)(ObjectPtr);
						
		/* Iterate over all entries of the modifier stack. */
		int ModStackIndex = 0;
		while (ModStackIndex < DerivedObjectPtr->NumModifiers())
		{
			/* Get current modifier. */
			Modifier* ModifierPtr = DerivedObjectPtr->GetModifier(ModStackIndex);

			/* Is this an type mod ? */
			if (ModifierPtr->ClassID() == type)
			{
				/* Yes -> Exit. */
				return ModifierPtr;
			}

			/* Next modifier stack entry. */
			ModStackIndex++;
		}
		ObjectPtr = DerivedObjectPtr->GetObjRef();
	}

	/* Not found. */
	return NULL;    
}

INode *
FindTopOfHierarchy( INode *node )
//find the highest node (just below scene root) in same heirarchy as given node
{
    while(node->GetParentNode() && !node->GetParentNode()->IsRootNode())
    {
        node = node->GetParentNode();
    }

    return node;
}

INode * FindFirstNodeInHierarchy( INode *node, FindFirstNodeInHierarchyCompareFn f )
//return the first node in a depth first search of the heirarchy that satifies
//the given compare function or 0 if nothing is found.
{
    if (f( node ))
    {
        return node;
    }

    int numChildren = node->NumberOfChildren();

    for (int i=0; i<numChildren; i++)
    {
        INode *foundNode = FindFirstNodeInHierarchy(node->GetChildNode(i), f );

        if (foundNode)
        {
            return foundNode;
        }
    }

    return 0;
}